import numpy as np


def hamming(vec1, vec2):
    """
    Compute the Hamming distance between two vectors.
    """
    vec1, vec2 = np.asarray(vec1), np.asarray(vec2)
    if len(vec1) != len(vec2):
        raise ValueError("Vectors must be of the same length.")
    return np.sum(vec1 != vec2)

def norm_hamming(vec1, vec2):
    """
    Compute the normalized Hamming distance between two vectors.
    """
    vec1, vec2 = np.asarray(vec1), np.asarray(vec2)
    if len(vec1) != len(vec2):
        raise ValueError("Vectors must be of the same length.")
    return np.sum(vec1 != vec2) / len(vec1)

def jaccard(vec1, vec2):
    """
    Compute the Jaccard similarity coefficient between two binary vectors.
    """
    vec1, vec2 = np.asarray(vec1, dtype=bool), np.asarray(vec2, dtype=bool)
    if len(vec1) != len(vec2):
        raise ValueError("Vectors must be of the same length.")
    if np.sum(vec1 | vec2) == 0:
        return 0
    M = np.sum(vec1 != vec2)
    return 2*M / (2*M +  np.sum(vec1 & vec2))

def rms_jaccard(vec1, vec2):
    """
    Compute the root-mean-square Jaccard similarity coefficient between two binary vectors.
    """
    vec1, vec2 = np.asarray(vec1, dtype=bool), np.asarray(vec2, dtype=bool)
    if len(vec1) != len(vec2):
        raise ValueError("Vectors must be of the same length.")
    a = np.sum(np.logical_and(vec1, ~vec2))
    b = np.sum(np.logical_and(~vec1, vec2))
    if np.sum(vec1 | vec2) == 0:
        return 0
    M = (a**2 + b**2)**0.5
    return 2*M / (2*M +  np.sum(vec1 & vec2))


def geom_jaccard(vec1, vec2):
    """
    Compute the geometric Jaccard similarity coefficient between two binary vectors.
    """
    vec1, vec2 = np.asarray(vec1, dtype=bool), np.asarray(vec2, dtype=bool)
    if len(vec1) != len(vec2):
        raise ValueError("Vectors must be of the same length.")
    a = np.sum(np.logical_and(vec1, ~vec2))
    b = np.sum(np.logical_and(~vec1, vec2))
    if np.sum(vec1 | vec2) == 0:
        return 0

    M = (a*b)**0.5
    return 2*M / (2*M +  np.sum(vec1 & vec2))

def geom_jaccard_plus(vec1, vec2):
    """
    Compute the geometric Jaccard similarity coefficient between two binary vectors.
    """
    vec1, vec2 = np.asarray(vec1, dtype=bool), np.asarray(vec2, dtype=bool)
    if len(vec1) != len(vec2):
        raise ValueError("Vectors must be of the same length.")
    a = np.sum(np.logical_and(vec1, ~vec2))
    b = np.sum(np.logical_and(~vec1, vec2))
    if np.sum(vec1 | vec2) == 0:
        return 0

    if (a==0 and b!=0) or (a!=0 and b==0):
        r = max(sum(vec1), sum(vec2)) / min(sum(vec1), sum(vec2))
        return 0.0001 * r

    M = (a*b)**0.5
    return 2*M / (2*M +  np.sum(vec1 & vec2))


def harmonic_jaccard(vec1, vec2):
    """
    Compute the root-mean-square Jaccard similarity coefficient between two binary vectors.
    """
    vec1, vec2 = np.asarray(vec1, dtype=bool), np.asarray(vec2, dtype=bool)
    if len(vec1) != len(vec2):
        raise ValueError("Vectors must be of the same length.")
    a = np.sum(np.logical_and(vec1, ~vec2))
    b = np.sum(np.logical_and(~vec1, vec2))
    if np.sum(vec1 | vec2) == 0:
        return 0
    if a==0 or b==0:
        return 1

    M = 1/(1/a + 1/b) /  np.sum(vec1 | vec2)
    return 2 * M / (2 * M + np.sum(vec1 & vec2))



def root_jaccard(vec1, vec2):
    """
    Compute the root Jaccard similarity coefficient between two binary vectors.
    """
    vec1, vec2 = np.asarray(vec1, dtype=bool), np.asarray(vec2, dtype=bool)
    if len(vec1) != len(vec2):
        raise ValueError("Vectors must be of the same length.")
    a = np.sum(np.logical_and(vec1, ~vec2))
    b = np.sum(np.logical_and(~vec1, vec2))
    if np.sum(vec1 | vec2) == 0:
        return 0
    M = (a**0.5 + b**0.5)**2
    return 2*M / (2*M +  np.sum(vec1 & vec2))


def geom_hamming(vec1, vec2):
    """
    Compute the geometric Hamming distance between two vectors.
    """
    vec1, vec2 = np.asarray(vec1, dtype=bool), np.asarray(vec2, dtype=bool)
    if len(vec1) != len(vec2):
        raise ValueError("Vectors must be of the same length.")
    a = np.sum(np.logical_and(vec1, ~vec2))
    b = np.sum(np.logical_and(~vec1, vec2))
    return 2*(a*b)**0.5 / len(vec1)


def geom_hamming_plus(vec1, vec2):
    """
    Compute the geometric Hamming distance between two vectors.
    """
    vec1, vec2 = np.asarray(vec1, dtype=bool), np.asarray(vec2, dtype=bool)
    if len(vec1) != len(vec2):
        raise ValueError("Vectors must be of the same length.")
    a = np.sum(np.logical_and(vec1, ~vec2))
    b = np.sum(np.logical_and(~vec1, vec2))

    if (a==0 and b!=0) or (a!=0 and b==0):
        r = max(sum(vec1), sum(vec2)) / min(sum(vec1), sum(vec2))
        return 0.0001 * r

    return 2*(a*b)**0.5 / len(vec1)

def rms_hamming(vec1, vec2):
    """
    Compute the geometric Hamming distance between two vectors.
    """
    vec1, vec2 = np.asarray(vec1, dtype=bool), np.asarray(vec2, dtype=bool)
    if len(vec1) != len(vec2):
        raise ValueError("Vectors must be of the same length.")
    a = np.sum(np.logical_and(vec1, ~vec2))
    b = np.sum(np.logical_and(~vec1, vec2))
    return (a ** 2 + b ** 2) ** 0.5 / len(vec1)
